%
%
%  Object Manager Function for Imaris 7.3.0
%
%  Copyright Bitplane AG 2011
%
%
%  Installation:
%
%  - Copy this file into the XTensions folder in the Imaris installation directory
%  - You will find this function in the Image Processing menu
%
%    <CustomTools>
%      <Menu>
%        <Item name="Open Object Manager" icon="Matlab" tooltip="Open object manager.">
%          <Command>MatlabXT::XTObjectManager(%i)</Command>
%        </Item>
%      </Menu>
%     <SurpassToolbar>
%        <Item name="Object Manager" icon="Matlab" tooltip="Open object manager.">
%          <Command>MatlabXT::XTObjectManager(%i)</Command>
%        </Item>
%     </SurpassToolbar>
%    </CustomTools>
% 
%
%  Description:
%   
%   Save/load/clear single imaris objects in .mat files.
%   Apply to Spots, Surfaces, Filaments and Groups.
%   Groups are saved with all their content.
%   Only first-level objects (mSurpassScene direct childs) can be saved.
%
%

function XTObjectManager(aImarisApplicationID)

if isempty(cell2mat(strfind(javaclasspath('-all'), 'ImarisLib.jar')))
  mlock
  javaaddpath ImarisLib.jar
  munlock
end

% persistent variables for callbacks 
persistent vImarisApplication vEditFileName

% button callbacks
if aImarisApplicationID==-1
    return;
elseif isa(aImarisApplicationID, 'numeric') && aImarisApplicationID==-5;
    Browse(vEditFileName);
    return;
elseif isa(aImarisApplicationID, 'numeric') && aImarisApplicationID<-1;
    Execute(vImarisApplication, vEditFileName, -aImarisApplicationID-1);
    return;
end

% connect to Imaris interface
if ~isa(aImarisApplicationID, 'Imaris.IApplicationPrxHelper')
  vImarisLib = ImarisLib;
  if ischar(aImarisApplicationID)
    aImarisApplicationID = round(str2double(aImarisApplicationID));
  end
  vImarisApplication = vImarisLib.GetApplication(aImarisApplicationID);
else
  vImarisApplication = aImarisApplicationID;
end

% create the window
uicontrol('Style', 'text', 'String', 'Imaris Object Manager', ...
    'FontName', 'Times New Roman', 'FontSize', 10, ...
    'Position', [10,70,170,20], 'DeleteFcn', 'XTObjectManager(-1)');
uicontrol('Style', 'pushbutton', 'String', 'Browse...', ...
    'Position', [210,70,80,20], 'Callback', 'XTObjectManager(-5)');
uicontrol('Style', 'text', 'String', 'File Name:', ...
    'FontName', 'Times New Roman', 'FontSize', 10, ...
    'Position', [10,50,70,20]);
vEditFileName = uicontrol('Style', 'edit', 'String', 'ImarisObjects.mat', ...
    'FontName', 'Times New Roman', 'FontSize', 10, ...
    'Position', [80,50,210,20]);

uicontrol('Style', 'pushbutton', 'String', 'Save', ...
    'Position', [10,10,80,30], 'Callback', 'XTObjectManager(-2)');
uicontrol('Style', 'pushbutton', 'String', 'Load', ...
    'Position', [110,10,80,30], 'Callback', 'XTObjectManager(-3)');
uicontrol('Style', 'pushbutton', 'String', 'Clear', ...
    'Position', [210,10,80,30], 'Callback', 'XTObjectManager(-4)');

vWindow = get(vEditFileName, 'Parent');
vWindowPosition = get(vWindow, 'Position');
vWindowPosition(3:4) = [300, 100];
set(vWindow, 'Position', vWindowPosition);

%----------------------------------------------%

function Browse(vEditFileName)
vFileName = get(vEditFileName,'String');
[vFileName, vPathName] = uiputfile({'*.mat','MAT-files (*.mat)'}, ...
    'Select the file', vFileName);
if ~isequal(vFileName,0)
    set(vEditFileName,'String',fullfile(vPathName,vFileName));
end


%----------------------------------------------%

function Execute(vImarisApplication, vEditFileName, aOperation, aParent, aParentName)
% aOperation = 1 save, 2 load, 3 clear

vStringOperations = {'save', 'load', 'clear'};
vStringTypes = {'Spots', 'Surfaces', 'Tracks', 'Filament', 'Group', 'Filaments'};
vFileName = get(vEditFileName, 'String');
vFactory = vImarisApplication.GetFactory;
vDataSet = vImarisApplication.GetDataSet;

if nargin==3
    vParent = vImarisApplication.GetSurpassScene;
    vParentName = '';
else 
    vParent = aParent;
    vParentName = aParentName;
end

vObjectNameString = [vParentName, 'vObjectsName'];
vObjectTypeString = [vParentName, 'vObjectsType'];

vObjectsName = {};
vObjectsType = [];
vObjectsIndex = [];
vNumberOfObjects = 0;

if aOperation==1
    % search the scene objects - Spots, Surfaces, Filaments and Groups
    for vChildIndex = 1:vParent.GetNumberOfChildren
        vDataItem = vParent.GetChild(vChildIndex - 1);
        vFound = false;
        if vFactory.IsSpots(vDataItem)
            vObjectsType(vNumberOfObjects+1) = 1;
            vFound = true;
        elseif vFactory.IsSurfaces(vDataItem)
            vObjectsType(vNumberOfObjects+1) = 2;
            vFound = true;
        elseif vFactory.IsDataContainer(vDataItem)
            vObjectsType(vNumberOfObjects+1) = 5;
            vFound = true;
        elseif vFactory.IsFilaments(vDataItem)
            vObjectsType(vNumberOfObjects+1) = 6;
            vFound = true;
        end
        
        if vFound
            vNumberOfObjects = vNumberOfObjects + 1;
            vObjectsName{vNumberOfObjects} = char(vDataItem.GetName);
            vObjectsIndex(vNumberOfObjects) = vChildIndex;
        end
    end
else
    % load the list of the saved objects
    if exist(vFileName, 'file')
        load(vFileName, vObjectNameString, vObjectTypeString);
        vObjectsName = eval(vObjectNameString);
        vObjectsType = eval(vObjectTypeString);
        
        vNumberOfObjects = length(vObjectsName);
    else
        vNumberOfObjects = 0;
    end
end

if vNumberOfObjects==0
    if nargin==3
        msgbox(['There is no object to ', vStringOperations{aOperation}, '!']);
    end
    return;
end

vObjectsCaption = vObjectsName;
for vObject = 1:vNumberOfObjects
    % Name [Type]
    vObjectsCaption{vObject} = [vObjectsName{vObject}, ' [', ...
        vStringTypes{vObjectsType(vObject)}, ']'];
end

if nargin==3
    [vObjectList, vOk] = listdlg('ListString', vObjectsCaption, ...
                           'SelectionMode','single',...
                           'ListSize',[250 150],...
                           'Name','Imaris Object Manager',...
                           'PromptString',{['Please select the object to ', ...
                           vStringOperations{aOperation}, ':']});
    if vOk<1, return; end;
    
else
    vObjectList = 1:vNumberOfObjects;
end

vObjectsName_ = vObjectsName;
vObjectsType_ = vObjectsType;

for vSelectedObject = vObjectList

    vName = vObjectsName_{vSelectedObject};
    vType = vObjectsType_(vSelectedObject);

    % remove invalid characters
    vName_ = vName; 
    for vLetter = 1:length(vName)
        vChar = vName(vLetter);
        if (vChar<'0' || vChar>'9') && (vChar<'a' || vChar>'z') && (vChar<'A' || vChar>'Z')
            vName_(vLetter) = '_';
        end
    end

    % names (that should be unique) of the variables in the file.mat
    vPrefix = [vParentName,'v',vName_,'_','0'+vType,'_'];
    vNameColorVisible = [vPrefix,'ColorVisible'];
    vNamePosition = [vPrefix,'Position'];
    vNameTime = [vPrefix,'Time'];
    vNameRadius = [vPrefix,'Radius'];
    vNameNormals = [vPrefix,'Normals'];
    vNameTriangles = [vPrefix,'Triangles'];
    vNameVertices = [vPrefix,'Vertices'];
    vNameEdges = [vPrefix,'Edges'];
    vNameSelection = [vPrefix,'Selection'];
    vNameTypes = [vPrefix,'Types'];
    vNameSpinesColor = [vPrefix,'SpinesColor'];
    vNameFilTrackEdges = [vPrefix,'FilTrackEdges'];
    vNameVertTrackEdges = [vPrefix,'VertTrackEdges'];
    vNameRoot = [vPrefix,'Root'];

    if aOperation==1 % save
        vDataItem = vParent.GetChild(vObjectsIndex(vSelectedObject) - 1);
        if vFactory.IsSpots(vDataItem)
            vObject = vFactory.ToSpots(vDataItem);
        elseif vFactory.IsSurfaces(vDataItem)
            vObject = vFactory.ToSurfaces(vDataItem);
        elseif vFactory.IsDataContainer(vDataItem)
            vObject = vFactory.ToDataContainer(vDataItem);
        elseif vFactory.IsFilaments(vDataItem)
            vObject = vFactory.ToFilaments(vDataItem);
        end

        if exist(vFileName, 'file')
            load(vFileName, vObjectNameString, vObjectTypeString);
            vObjectsName = eval(vObjectNameString);
            vObjectsType = eval(vObjectTypeString);

            % if the variable is already saved, their values are overwritten
            % but no new object is added
            vFound = false;
            vNumberOfObjects = length(vObjectsName);
            for vIndex = 1:vNumberOfObjects
                if strcmp(vName,vObjectsName{vIndex}) && vType==vObjectsType(vIndex)
                    vFound = true;
                    break;
                end
            end
            if vFound
                vAnswer = questdlg(['Overwrite ',vObjectsCaption{vSelectedObject},'?'], ...
                    'Imaris Object Manager', 'Yes', 'No', 'Yes');
                if ~strcmp(vAnswer,'Yes')
                    return;
                elseif vType==5
                    % clear all the children of the group
                    Execute(vImarisApplication, vEditFileName, 3, vObject, vPrefix);
                end
            else
                vObjectsName{vNumberOfObjects+1} = vName;
                vObjectsType(vNumberOfObjects+1) = vType;
            end
            eval([vObjectNameString,' = vObjectsName;']);
            eval([vObjectTypeString,' = vObjectsType;']);
            save(vFileName, vObjectNameString, vObjectTypeString, '-append');
        else
            vObjectsName = {vName};
            vObjectsType = vType;
            save(vFileName, 'vObjectsName', 'vObjectsType');   
        end

        eval([vNameColorVisible, ' = [vObject.GetColorRGBA, vObject.GetVisible];']);
        save(vFileName, vNameColorVisible, '-append');

        if vType==1 % spots			  		  
            eval([vNamePosition,' = vObject.GetPositionsXYZ;']);
            eval([vNameTime,' = vObject.GetIndicesT;']);
            eval([vNameRadius,' = vObject.GetRadiiXYZ;']);			  
            save(vFileName, vNamePosition, vNameRadius, '-append');
        elseif vType==2 % surfaces
            vNumSurfs = vObject.GetNumberOfSurfaces;
            vTimes = 1:vNumSurfs;
            for vIndex = 0:vNumSurfs-1
                vNum = sprintf('_%d', vIndex);
                eval([vNameVertices,vNum,' = vObject.GetVertices(vIndex);']);
                eval([vNameTriangles,vNum,' = vObject.GetTriangles(vIndex);']);
                eval([vNameNormals,vNum,' = vObject.GetNormals(vIndex);']);
                save(vFileName, [vNameVertices,vNum], ...
                  [vNameTriangles,vNum], [vNameNormals,vNum], '-append');
                vTimes(vIndex+1) = vObject.GetTimeIndex(vIndex);
            end
            eval([vNameTime,' = vTimes;']);
            save(vFileName, vNameTime, '-append');
        elseif vType==5 % data container
            eval([vNameTime,' = 0;']);
            % save all the childrens  
            eval([vPrefix,'vObjectsName',' = {};']);
            eval([vPrefix,'vObjectsType',' = [];']);
            save(vFileName,[vPrefix,'vObjectsName'],'-append');
            save(vFileName,[vPrefix,'vObjectsType'],'-append');
            Execute(vImarisApplication, vEditFileName, 1, vObject, vPrefix)
        elseif vType==6 % filaments
            vNumFils = vObject.GetNumberOfFilaments;
            vTimes = 1:vNumFils;
            for vIndex = 0:vNumFils-1
                vNum = sprintf('_%d', vIndex);
                eval([vNamePosition,vNum,' = vObject.GetPositionsXYZ(vIndex);']);
                eval([vNameRadius,vNum,' = vObject.GetRadii(vIndex);']);
                eval([vNameTypes,vNum,' = vObject.GetTypes(vIndex);']);
                eval([vNameEdges,vNum,' = vObject.GetEdges(vIndex);']);
                eval([vNameSelection,vNum,' = vObject.GetSelectedPositionIndices(vIndex);']);
                eval([vNameRoot,vNum,' = vObject.GetBeginningVertexIndex(vIndex);']);
                save(vFileName, [vNamePosition,vNum], [vNameRadius,vNum], ...
                  [vNameTypes,vNum], [vNameEdges,vNum], ...
                  [vNameSelection,vNum], [vNameRoot,vNum], '-append');
                vTimes(vIndex+1) = vObject.GetTimeIndex(vIndex);
            end
            eval([vNameTime,' = vTimes;']);
            eval([vNameSpinesColor, ' = vObject.GetColorSpinesRGBA;']);
            eval([vNameFilTrackEdges,' = vObject.GetFilamentTrackEdges;']);
            eval([vNameVertTrackEdges,' = vObject.GetVertexTrackEdges;']);
            save(vFileName, vNameTime, vNameSpinesColor, vNameFilTrackEdges, ...
              vNameVertTrackEdges, '-append');
        end
        
        if vType == 1 || vType == 2
            % features common to spots and surfaces
            eval([vNameEdges,' = vObject.GetTrackEdges;']);
            eval([vNameSelection,' = vObject.GetSelectedIndices;']);
            save(vFileName, vNameEdges, vNameSelection, '-append');
        end

        % time is required for all oject types (please set dummy 0 if not available)
        save(vFileName, vNameTime, '-append');

    elseif aOperation==2 % load

        load(vFileName, vNameTime);
        vTime = max(eval(vNameTime))+1;
        if vTime>vDataSet.GetSizeT
            vAnswer = questdlg(sprintf(['%s is composited by %i frames. ', ...
                'Resizing the DataSet could cause memory problems. Procede anyway?'], ...
                vObjectsCaption{vSelectedObject},vTime), ...
                'Imaris Object Manager', 'Yes', 'No', 'Yes');
            if ~strcmp(vAnswer,'Yes')
                return
            end

           vDataSet.SetSizeT(vTime);
        end

        if vType==1 % spots
            load(vFileName, vNamePosition, vNameRadius);        
            vObject = vFactory.CreateSpots;
			% The radius is after overwritten by the SetRadiiXYZ function
			% So the radius can be set just by using the radius of one axis, Here the X axis is used
            eval(['vObject.Set(',vNamePosition,',',vNameTime,',',vNameRadius,'(:,1));']);
            eval(['vObject.SetRadiiXYZ(',vNameRadius,');']);
        elseif vType==2 % surfaces
            vObject = vFactory.CreateSurfaces;
            vTimes = eval(vNameTime);
            vNumSurfs = numel(vTimes);
            for vIndex = 0:vNumSurfs-1
              vNum = sprintf('_%d', vIndex);
              load(vFileName, [vNameVertices,vNum], ...
                  [vNameTriangles,vNum], [vNameNormals,vNum]);
              eval(['vObject.AddSurface(',vNameVertices,vNum,',', ...
                  vNameTriangles,vNum,',',vNameNormals,vNum,',', ...
                  sprintf('%d', vTimes(vIndex+1)),');']);
            end
        elseif vType==4 % filament
            load(vFileName, vNamePosition, vNameRadius, vNameEdges);
            vObject = vFactory.CreateFilaments;
            eval(['vObject.AddFilament(',vNamePosition,',',vNameRadius,',', ...
              zeros(size(eval(vNamePosition),1),1),',',vNameEdges,',',vNameTime,');']);    
        elseif vType==5 % data container
            vObject = vFactory.CreateDataContainer;
            Execute(vImarisApplication, vEditFileName, 2, vObject, vPrefix);
        elseif vType==6 % filaments
            vObject = vFactory.CreateFilaments;
            vTimes = eval(vNameTime);
            vNumFils = numel(vTimes);
            for vIndex = 0:vNumFils-1
              vNum = sprintf('_%d', vIndex);
              load(vFileName, [vNamePosition,vNum], [vNameRadius,vNum], ...
                  [vNameTypes,vNum], [vNameEdges,vNum]);
              eval(['vObject.AddFilament(',vNamePosition,vNum,',', ...
                  vNameRadius,vNum,',',vNameTypes,vNum,',', ...
                  vNameEdges,vNum,',', ...
                  sprintf('%d', vTimes(vIndex+1)),');']);
            end
            for vIndex = 0:vNumFils-1
              vNum = sprintf('_%d', vIndex);
              load(vFileName, [vNameSelection,vNum], [vNameRoot,vNum]);
              eval(['vObject.SetSelectedPositionIndices(vIndex,',vNameSelection,vNum,');']);
              eval(['vObject.SetBeginningVertexIndex(vIndex,',vNameRoot,vNum,');']);
            end
            load(vFileName, vNameSpinesColor, vNameFilTrackEdges, ...
              vNameVertTrackEdges);
            eval(['vObject.SetColorSpinesRGBA(',vNameSpinesColor,');']);
            eval(['vObject.SetFilamentTrackEdges(',vNameFilTrackEdges,');']);
            eval(['vObject.SetVertexTrackEdges(',vNameVertTrackEdges,');']);
        end

        if vType == 1 || vType == 2
            % features common to spots and surfaces
            load(vFileName, vNameEdges, vNameSelection);
            eval(['vObject.SetTrackEdges(',vNameEdges,');']);    
            eval(['vObject.SetSelectedIndices(',vNameSelection,');']);    
        end

        load(vFileName, vNameColorVisible);
        eval(['vObject.SetColorRGBA(',vNameColorVisible,'(1));']);
        eval(['vObject.SetVisible(',vNameColorVisible,'(2));']);

        vObject.SetName(vName);
        vParent.AddChild(vObject, -1);

    else % clear

        % copy the data into a struct
        % deletes the original file
        % rebuild the right file

        if nargin==3
            vAnswer = questdlg(['Remove ',vObjectsCaption{vSelectedObject},' from the memory?'], ...
                'Imaris Object Manager', 'Yes', 'No', 'Yes');
            if ~strcmp(vAnswer,'Yes')
                return;
            end
        end

        vFile = load(vFileName);
        vFields = fieldnames(vFile);
        if nargin==3
            % remove the name and the type from the lists
            vRemaining = [1:vSelectedObject-1,vSelectedObject+1:vNumberOfObjects];
            vObjectsName = vObjectsName(vRemaining);
            vObjectsType = vObjectsType(vRemaining);
        end
        save(vFileName, 'vObjectsName', 'vObjectsType');

        vNumberOfLetters = length(vPrefix);
        for vVariable = 1:length(vFields)
            vString = vFields{vVariable};
            if ~strcmp(vString,'vObjectsName') && ~strcmp(vString,'vObjectsType') && ...
                    (length(vString)<vNumberOfLetters || ...
                    ~strcmp(vPrefix,vString(1:vNumberOfLetters)))
                eval([vString,' = vFile.(vString);']);
                save(vFileName,vString,'-append');
                clear(vString);
            end
        end
    end
end

